//
//  WXKPhone.m
//  KyoponUtilities2
//
//  Created by FUJIDANA on 06/07/20.
//  Copyright 2006 FUJIDANA. All rights reserved.
//
//
//  Some part of this source code is derived from AirHPhone.m, 
//  written by raktajino in AH-K3001V Bookmark Utility and distributed
//  under BSD license.
//  Communicational protocol to AH-K3001V was referred to ``AirH" 
//  AH-K3001 station'' (http://www.softclub.jp/zoro/ce/usb.html), written
//  by Zoroyoshi and distributed under BSD license.
//  See the following about copyrights.
//
//
//  Created by raktajino on Thu Jun 17 2004.
//  Copyright (c) 2004 raktajino. All rights reserved.
//
//		Copyright (c) 04 Zoroyoshi, Japan
//		All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "WXKPhone.h"
#import "WXKError.h"
#import "WXKProgressing.h"

#import <stdio.h>
#import <unistd.h>
#import <errno.h>
#import <fcntl.h>
#import <sys/ioctl.h>

#define kOpeningRetryCount			8		// max retry number on open/log-in.
#define kOpeningRetryInterval		0.25	// retry interval on open/log-in. (sec)
#define kRequireDriverMajorVersion	1		// required major verision number of AH-K3001V USB Driver
#define kRequireDriverMinorVersion	2		// required minor verision number of AH-K3001V USB Driver

#define kReadBufferSize				4096	// preferred buffer size for reading
#define kWriteBufferSize			4096	// preferred buffer size for writing


#pragma mark -


@interface WXKPhone (Private)

- (BOOL)openError:(NSError **)errorPtr;
- (BOOL)closeError:(NSError **)errorPtr;
- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(const char *)parameter error:(NSError **)errorPtr;
- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(const UInt8 *)parameter length:(SInt32)length error:(NSError **)errorPtr;
- (BOOL)sendRawBytes:(const UInt8 *)buffer length:(SInt32)length error:(NSError **)errorPtr;
//- (SInt32)receiveBytes:(UInt8 *)buffer maxLength:(SInt32)bufferSize additionally:(BOOL)hasAdditionallydata error:(NSError **)errorPtr;
- (BOOL)receiveData:(NSData **)data error:(NSError **)errorPtr;
- (SInt32)receiveRawBytes:(UInt8 *)buffer maxLength:(SInt32)bufferSize error:(NSError **)errorPtr;
- (BOOL)loginWithPassword:(NSString *)password error:(NSError **)errorPtr;
- (BOOL)logoutError:(NSError **)errorPtr;

@end


#pragma mark -


@implementation WXKPhone


#pragma mark public class methods

// --- check whtether the driver has been installed. Return yes if the driver exists.

+ (BOOL)existsDriver
{
	NSArray *paths = NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, NSSystemDomainMask, YES);
	if (paths == nil || [paths count] == 0) return NO;
	
	NSString *driverPath = [[[paths lastObject] stringByAppendingPathComponent:@"Extensions"] stringByAppendingPathComponent:@"AHK3001VDriver.kext"];
	NSBundle *driverBundle = [NSBundle bundleWithPath:driverPath];
	if (driverBundle == nil) return NO;
	
	NSString *versionString = [driverBundle objectForInfoDictionaryKey: @"CFBundleShortVersionString"];
	if (versionString == nil) return NO;
	
	NSArray *versionArray = [versionString componentsSeparatedByString:@"."];
	if ([versionArray count] < 2) return NO;
	
	SInt32 currentMajorVersion = [[versionArray objectAtIndex:0] intValue];
	SInt32 currentMinorVersion = [[versionArray objectAtIndex:1] intValue];
	
	if (currentMajorVersion > kRequireDriverMajorVersion) {
		return YES;
	} else if (currentMajorVersion == kRequireDriverMajorVersion && currentMinorVersion >= kRequireDriverMinorVersion) {
		return YES;
	}
	return NO;
}

// --- get name of required driver

+ (NSString *)requiredDriverName
{
	return [NSString stringWithFormat:@"AH-K3001V USB Driver %d.%d", kRequireDriverMajorVersion, kRequireDriverMinorVersion];
}

// --- parse data 

+ (BOOL)getParsedDataFields:(NSData **)dataFields fieldNumber:(unsigned)fieldNum fromData:(NSData *)rawData
{
	const UInt8 *bytes = [rawData bytes];
	SInt32 readLength = 13;
	
	while (readLength < [rawData length] && bytes[readLength] == 0x1e) {
//		SInt32  fieldLength = bytes[readLength + 3] + (bytes[readLength + 4] << 8) + (bytes[readLength + 5] << 16) + (bytes[readLength + 6] << 24);
		SInt32 *fieldLengthPointer = (SInt32 *)&bytes[readLength + 3];
		SInt32  fieldLength = NSSwapLittleLongToHost(*fieldLengthPointer);
		SInt32  fieldIndex  = bytes[readLength + 1] - 1;
		if (fieldIndex >= fieldNum) {
			return NO;
		}
		
		if (fieldLength == 0x01000000) {
			fieldLength = bytes[readLength + 2];
		}
		readLength += 7;
		
		dataFields[fieldIndex] = [NSData dataWithBytes:bytes + readLength length:fieldLength];
		
		readLength += fieldLength;
	}
	
	if (readLength == [rawData length]) {
		return YES;
	} else if (readLength > [rawData length]) {
		NSLog(@"ERROR. POINTER IS OVER THE DATA LENGTH");
		return NO;
	} else {
		NSLog(@"ERROR. Data field is not separated with correct code.");
		return NO;
	}
}

#pragma mark init and dealloc

// --- initialize object

- (id)init
{
	self = [super init];
	if (self != nil) {
		fileDescriptor = -1;
		loggedin       = NO;
	}
	return self;
}

// --- deallocate object

- (void)dealloc
{
	[self closeError:NULL];
	
	[super dealloc];
}


#pragma mark public methods

- (BOOL)receiveAllItemsOfDataType:(WXKPhoneDataType)dataType
					intoDataArray:(NSArray **)dataArray
					 withPassword:(NSString *)password
						 progress:(id <WXKProgressing>)progress
							error:(NSError **)errorPtr
{
	NSData *receivedData;
	const UInt8 *bytes;
	
	NSAutoreleasePool *pool = nil;
	NSMutableArray *mutableDataArray;
	
	// begin progress and login
	[progress beginProgressWithMessage:NSLocalizedString(@"Opening Port...", @"phone.message.openPort")];
	if ([self openError:errorPtr] == NO) goto ERROR;
	
 	// send password
	[progress updateDoubleValue:-1.0 message:NSLocalizedString(@"Sending Security Code...", @"phone.message.password")];
	if ([self loginWithPassword:password error:errorPtr] == NO) goto ERROR;
	
	// count number of items
	if ([self sendCommand:2 subCommand:(dataType << 8) + 1 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ([self receiveData:&receivedData error:errorPtr] == NO) goto ERROR;
	bytes = [receivedData bytes];
	const int count = atoi((const char *)bytes + 12);
	
	mutableDataArray = [NSMutableArray arrayWithCapacity:count];
	
	[progress startDeterminateProgressWithMaxValue:count
										   message:NSLocalizedString(@"Receiving...", @"phone.message.readItems")];
	
	if (count > 0) {
		
		// begin to read items.
		if ([self sendCommand:3 subCommand:(dataType << 8) + 2 parameter:"" error:errorPtr] == NO) goto ERROR;
		
		int n;
		for (n = 0; n < count; n++) {
			
			pool = [[NSAutoreleasePool alloc] init];
			
			// Now safe to abort communication, so check whether user requested to abort progress
			if ([progress isRequestedToAbort]) {
				if (errorPtr) {
					NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Reading was interrupted by user.", @"error.userCancelledWhileReading.description")
																		 forKey:NSLocalizedDescriptionKey];
					*errorPtr = [NSError errorWithDomain:WXKErrorDomain
													code:WXKPhoneUserCancelledError
												userInfo:userInfo];
				}
				goto ERROR;
			}
			
			// Read an item.
			if ([self receiveData:&receivedData error:errorPtr] == NO) goto ERROR;
			bytes = [receivedData bytes];
			
			if (bytes[3] != 0x03 || bytes[9] != dataType || bytes[11] != 0x02) {
				if (errorPtr) {
					NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Illegal format data were found.", @"error.inapplicableDataReceived.description")
																		 forKey:NSLocalizedDescriptionKey];
					*errorPtr = [NSError errorWithDomain:WXKErrorDomain
													code:WXKPhoneInapplicableDataReceivedError
												userInfo:userInfo];
				}
				goto ERROR;
			}
			
			// Notify that reading has been finished
			if ([self sendCommand:0x10 subCommand:0x0910 parameter:"0" error:errorPtr] == NO) goto ERROR;
			
//			SInt32 dataLength = sizeof(buffer) > restBytes ? restBytes : sizeof(buffer);
//			[mutableDataArray addObject:[NSData dataWithBytes:buffer length:dataLength]];
			
			[mutableDataArray addObject:receivedData];
			
			// refresh progress indicator
			[progress incrementProgressBarBy:1];
			
			[pool release], pool = nil;
		}
	}
	[progress endProgress];
	if ([self closeError:errorPtr] == NO) return NO;
	
	if (dataArray != NULL) {
		*dataArray = mutableDataArray;
	}
	
	return YES;
	
ERROR:
	// Do not release pool! If you release this autorelease pool, the error 
	// object is also released before a receiver of this method handles the error.
	// if (pool) [pool release]; 
	
	[progress endProgress];
	[self closeError:NULL];
	return NO;
}

- (BOOL)writeAllItemsOfDataType:(WXKPhoneDataType)dataType
					  dataArray:(NSArray *)dataArray
				   withPassword:(NSString *)password
					   progress:(id <WXKProgressing>)progress
						  error:(NSError **)errorPtr
{
	// check number of items is under the limitation or not.
	const SInt32 count = [dataArray count];
	
	BOOL  isInAppendMode = NO;
	
	NSData *receivedData;
	const UInt8 *bytes;
	
	SInt32 n;
	
	// begin progress
	[progress beginProgressWithMessage:NSLocalizedString(@"Sending Security Code...", @"phone.message.password")];
	
	// log in and send password
	if ([self openError:errorPtr] == NO) goto ERROR;
	if ([self loginWithPassword:password error:errorPtr] == NO) goto ERROR;
	
	// remove all items in cellphone
	if ([self sendCommand:3 subCommand:(dataType << 8) + 4 parameter:"13" error:errorPtr] == NO) goto ERROR;
	if ([self receiveData:NULL error:errorPtr] == NO) goto ERROR;
	
	// begin to send all items
	if ([self sendCommand:3 subCommand:(dataType << 8) + 5 parameter:"" error:errorPtr] == NO) goto ERROR;
	if ([self receiveData:NULL error:errorPtr] == NO) goto ERROR;
	isInAppendMode = YES;
	
	[progress startDeterminateProgressWithMaxValue:count message:NSLocalizedString(@"Sending...", @"phone.message.send")];
	
	for (n = 0; n < count; n++) {
		if ([progress isRequestedToAbort]) {
			if (errorPtr) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Writing is interrupted by user. The database of your cellular phone is not incomplete. Remind that if you read data from your cellular phone the database in your computer might be lost.", @"error.userCancelledWhileWriting.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:WXKErrorDomain
												code:WXKPhoneUserCancelledError
											userInfo:userInfo];
			}
			goto ERROR;
		}
		
		NSData *data = [dataArray objectAtIndex:n];
		
		// send a item.
		if ([self sendCommand:3 subCommand:(dataType << 8) + 3 parameter:[data bytes] length:[data length] error:errorPtr] == NO) goto ERROR;
		if ([self receiveData:&receivedData error:errorPtr] == NO) goto ERROR;
		
		bytes = [receivedData bytes];
		
		if(strncmp((const char *)bytes + 12, "0", 1)) {
			if (errorPtr) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The AIR-EDGE PHONE didn't return the success code.", @"error.failToSend.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:WXKErrorDomain
												code:WXKPhoneMalformedItemSentError
											userInfo:userInfo];
			}
			goto ERROR;
//			[NSException raise:kAirHPhoneException format:@"The AIR-EDGE PHONE didn't return the success code."];
		}
		// refresh progress status
		[progress incrementProgressBarBy:1];
	}
	
	if (isInAppendMode) {
		// finish sending items
		if ([self sendCommand:3 subCommand:(dataType << 8) + 6 parameter:"" error:errorPtr] == NO) goto ERROR;
		if ([self receiveData:NULL error:errorPtr] == NO) goto ERROR;
	}
	
	[progress endProgress];
	[self closeError:errorPtr];
	return YES;
	
ERROR:
	if (isInAppendMode) {
		// stop receiving items
		[self sendCommand:3 subCommand:(dataType << 8) + 6 parameter:"" error:NULL];
		[self receiveData:NULL error:NULL];
	}
	[progress endProgress];
	[self closeError:NULL];
	return NO;
}

@end


#pragma mark -


@implementation WXKPhone (Private)

#pragma mark private methods

// --- open connection to cellphone

- (BOOL)openError:(NSError **)errorPtr
{
	// open serial port
	int n = 0;
	int errorNum = 0;
	while (true) {
		fileDescriptor = open("/dev/cu.ah-k3001v.data", O_RDWR | O_NOCTTY | O_NONBLOCK);
		if (fileDescriptor != -1) {
			// Port was successfully opened.
			break;
		} else {
			// Port was not opened. Retry or return error.
			errorNum = errno;
			if (errorNum == ENOENT) {
				if (n++ < kOpeningRetryCount) {
					[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:kOpeningRetryInterval]]; 
					continue;
				}
				// @"The AIR-EDGE PHONE wasn't connected."
				if (errorPtr != NULL) {
					NSError *underError = [NSError errorWithDomain:NSPOSIXErrorDomain
															  code:errorNum
														  userInfo:nil];
					NSString *locDesc = NSLocalizedString(@"The AIR-EDGE PHONE wasn't connected.", @"error.phoneConnection.description");
					NSDictionary *userInfo = [NSDictionary dictionaryWithObjectsAndKeys:locDesc,
						NSLocalizedDescriptionKey,
						underError,
						NSUnderlyingErrorKey,
						nil];
					*errorPtr = [NSError errorWithDomain:WXKErrorDomain 
													code:WXKPhoneConnectionError
												userInfo:userInfo];
				}
				return NO;
			} else {
				// @"Couldn't open a port for the AIR-EDGE PHONE."
				if (errorPtr != NULL) {
					NSError *underError = [NSError errorWithDomain:NSPOSIXErrorDomain
															  code:errorNum
														  userInfo:nil];
					NSString *locDesc = NSLocalizedString(@"Couldn't open a port for the AIR-EDGE PHONE.", @"error.openPort.description");
					NSDictionary *userInfo = [NSDictionary dictionaryWithObjectsAndKeys:locDesc,
						NSLocalizedDescriptionKey,
						underError,
						NSUnderlyingErrorKey,
						nil];
					*errorPtr = [NSError errorWithDomain:WXKErrorDomain
													code:WXKPhoneOpenPortError
												userInfo:userInfo];
				}
				return NO;
			}
		}
	}
	
	// set the serial port in exclusive use mode
	if (ioctl(fileDescriptor, TIOCEXCL) == -1) {
		errorNum = errno;
		goto ERROR;
	}
	
	if (fcntl(fileDescriptor, F_SETFL, 0) == -1) {
		errorNum = errno;
		goto ERROR;
	}
	
	// get and store the current attribute of the serial port
	if (tcgetattr(fileDescriptor, &originalTTYAttrs) == -1) {
		errorNum = errno;
		goto ERROR;
	}
	
	// change the attribute of the serial port
	struct termios options = originalTTYAttrs;
	options.c_cflag = CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
	options.c_iflag = IGNBRK | IGNPAR;
	options.c_oflag = 0;
	options.c_lflag = 0;
	for (n = 0; n < NCCS; n++) {
		options.c_cc[n] = 0;
	}
	options.c_cc[VMIN]  = 1;
	options.c_cc[VTIME] = 0;
	if (tcsetattr(fileDescriptor, TCSANOW, &options) == -1) {
		errorNum = errno;
		goto ERROR;
	}
	
	loggedin = NO;
	return YES;
	
ERROR:
	// @"Couldn't set the serial port options."
	if (fileDescriptor != -1) {
		close(fileDescriptor);
		fileDescriptor = -1;
	}
	
	if (errorPtr != NULL) {
		NSError *underError = [NSError errorWithDomain:NSPOSIXErrorDomain
												  code:errorNum
											  userInfo:nil];
		NSString *locDesc = NSLocalizedString(@"Couldn't set the serial port options.", @"error.setSerialPortOptions.description");
		NSDictionary *userInfo = [NSDictionary dictionaryWithObjectsAndKeys:locDesc,
			NSLocalizedDescriptionKey,
			underError,
			NSUnderlyingErrorKey,
			nil];
		*errorPtr = [NSError errorWithDomain:WXKErrorDomain 
										code:WXKPhoneConfigurePortError
									userInfo:userInfo];
	}
	return NO;
}

// --- close connection to a cellphone

- (BOOL)closeError:(NSError **)errorPtr
{
	BOOL flag = YES;
	if (fileDescriptor != -1) {
		flag = [self logoutError:errorPtr];
		tcdrain(fileDescriptor);
		tcsetattr(fileDescriptor, TCSANOW, &originalTTYAttrs);
		close(fileDescriptor);
		fileDescriptor = -1;
	}
	return flag;
}

// --- send command to a cellphone

- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(const char *)parameter error:(NSError **)errorPtr
{
	SInt32 length = strlen(parameter);
	return [self sendCommand:a subCommand:b parameter:(const UInt8 *)parameter length:length error:errorPtr];
}


- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(const UInt8 *)parameter length:(SInt32)length error:(NSError **)errorPtr
{
	UInt8 buf[12] = {
		0xe1, 0x01, (UInt8)(a >> 8), (UInt8)a,
		0, 0, 0, 0,
		0, (UInt8)(b >> 8), 0, (UInt8)b
	};
	
	const SInt32 lenWithHeader = length + 4;
	buf[4] = (UInt8) lenWithHeader;
	buf[5] = (UInt8)(lenWithHeader >> 8 );
	buf[6] = (UInt8)(lenWithHeader >> 16);
	buf[7] = (UInt8)(lenWithHeader >> 24);
	
	if ([self sendRawBytes:buf length:sizeof(buf) error:errorPtr] == NO) return NO;
	
	if(parameter != nil) {
		if ([self sendRawBytes:parameter length:length error:errorPtr] == NO) return NO;
	}
	return YES;
}

// --- receive data from a cellphone

- (SInt32)receiveBytes:(UInt8 *)buffer maxLength:(SInt32)bufferSize additionally:(BOOL)hasAdditionallyData error:(NSError **)errorPtr
{
	// write end mark on buffer
	buffer[12] = 0xff;
	buffer[13] = 0x00;
	
	int bufferLength = 0;
	while (bufferLength < 12) {
		// receive data
		SInt32 numBytes = [self receiveRawBytes:(buffer + bufferLength) maxLength:(12 - bufferLength) error:errorPtr];
		if (numBytes == -1) {
			return -1;
		}
		
		bufferLength += numBytes;
		
		// seek header mark
		int n;
		for (n = 0; n < bufferLength - 1; n++) {
			if (buffer[n] == 0xe1 && buffer[n + 1] == 0x02) {
				break;
			}
		}
		
		// if header mark is found, move buffer data so that header mark becomes the beginning of buffer
		if (n > 0) {
			bufferLength -= n;
			memmove(buffer, buffer + n, bufferLength);
		}
	}
	
	const int len = buffer[4] + (((SInt32)buffer[5]) << 8) + (((SInt32)buffer[6]) << 16) + (((SInt32)buffer[7]) << 24) + 8;
	
	if (hasAdditionallyData) {
		if(len < 12) {
			if (errorPtr != NULL) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"An illegal length data was received.", @"phone.InapplicableDataReceivedError.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:WXKErrorDomain
												code:WXKPhoneInapplicableDataLengthReceivedError
											userInfo:userInfo];
			}
			return -1;
		}
		
		const int size = (len > bufferSize) ? bufferSize : len;
		bufferLength = 12;
		while (bufferLength < size) {
			SInt32 numBytes = [self receiveRawBytes:buffer + bufferLength maxLength:size - bufferLength error:errorPtr];
			if (numBytes == -1) {
				return -1;
			}
			bufferLength += numBytes;
		}
	}
	return len;
}


// --- receive data from a cellphone

- (BOOL)receiveData:(NSData **)data error:(NSError **)errorPtr
{
	UInt8 buffer[kReadBufferSize];
	
	// write end mark on buffer
	buffer[12] = 0xff;
	buffer[13] = 0x00;
	
	int readLength = 0;
	while (readLength < 12) {
		// receive data
		SInt32 numBytes = [self receiveRawBytes:(buffer + readLength) maxLength:(12 - readLength) error:errorPtr];
		if (numBytes == -1) {
			return NO;
		}
		
		readLength += numBytes;
		
		// seek header mark
		int n;
		for (n = 0; n < readLength - 1; n++) {
			if (buffer[n] == 0xe1 && buffer[n + 1] == 0x02) {
				break;
			}
		}
		
		// if header mark is found, move buffer data so that header mark becomes the beginning of buffer
		if (n > 0) {
			readLength -= n;
			memmove(buffer, buffer + n, readLength);
		}
	}
	
	const int dataLength = buffer[4] + (((SInt32)buffer[5]) << 8) + (((SInt32)buffer[6]) << 16) + (((SInt32)buffer[7]) << 24) + 8;
	
	NSMutableData *mutableData = [NSMutableData dataWithCapacity:dataLength];
	[mutableData appendBytes:buffer length:12];
	
//	if (hasAdditionallyData) {
	
	if (dataLength < 12) {
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"An illegal length data was received.", @"phone.InapplicableDataReceivedError.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneInapplicableDataLengthReceivedError
										userInfo:userInfo];
		}
		return NO;
	}
	
	int currentReadLength = ((dataLength - [mutableData length]) > sizeof(buffer)) ? sizeof(buffer) : (dataLength - [mutableData length]);
	while ([mutableData length] < dataLength) {
		SInt32 numBytes = [self receiveRawBytes:buffer maxLength:currentReadLength error:errorPtr];
		if (numBytes == -1) {
			return NO;
		}
		[mutableData appendBytes:buffer length:numBytes];
	}
	
	if ([mutableData length] != dataLength) {
		NSLog(@"Illegal data length");
	}
	
	if (data != NULL) {
		*data = [NSData dataWithData:mutableData];
//		*data = [NSData dataWithBytes:buffer length:readLength];
	}
	
//	}
	return YES;
}

// --- Send data to a cellphone without modification.

- (BOOL)sendRawBytes:(const UInt8 *)buffer length:(SInt32)length error:(NSError **)errorPtr
{
	SInt32 numBytes = write(fileDescriptor, buffer, length);
	if (numBytes == -1) {
		if (errorPtr != NULL) {
			NSError *underError = [NSError errorWithDomain:NSPOSIXErrorDomain
													  code:errno
												  userInfo:nil];
			NSString *locDesc = NSLocalizedString(@"Couldn't write the serial port.", @"error.sendBytes.description");
			NSDictionary *userInfo = [NSDictionary dictionaryWithObjectsAndKeys:locDesc,
				NSLocalizedDescriptionKey,
				underError,
				NSUnderlyingErrorKey,
				nil];
			
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneWriteSerialPortError
										userInfo:userInfo];
		}
		return NO;
		
	} else if (numBytes != length) {
		if (errorPtr != NULL) {
			NSString *locDesc = NSLocalizedString(@"Couldn't write the serial port.", @"error.sendBytes.description");
			NSDictionary *userInfo = [NSDictionary dictionaryWithObjectsAndKeys:locDesc,
				NSLocalizedDescriptionKey,
				nil];
			
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneWriteSerialPortError
										userInfo:userInfo];
		}
		return NO;
	}
	return YES;
}

// --- receive data from a cellphone without modification 

- (SInt32)receiveRawBytes:(UInt8 *)buffer maxLength:(SInt32)bufferSize error:(NSError **)errorPtr
{
	SInt32 numBytes = read(fileDescriptor, buffer, bufferSize);
	
	if (numBytes == -1) {
		if (errorPtr != NULL) {
			NSError *underError = [NSError errorWithDomain:NSPOSIXErrorDomain
													  code:errno
												  userInfo:nil];
			NSDictionary *userInfo = [NSDictionary dictionaryWithObjectsAndKeys:NSLocalizedString(@"Couldn't read the serial port.", @"error.receiveBytes.description"),
				NSLocalizedDescriptionKey, 
				underError,
				NSUnderlyingErrorKey, 
				nil];
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneReadSerialPortError
										userInfo:userInfo];
		}
		return -1;
	}
	return numBytes;
}


// --- login to a cellphone

- (BOOL)loginWithPassword:(NSString *)password error:(NSError **)errorPtr
{
	UInt8 buffer[kReadBufferSize];
	
	// login
	int n = 0;
	while (true) {
		if ([self sendCommand:0 subCommand:0x0101 parameter:"1" error:errorPtr] == NO) {
			return NO;
		}
		if ([self receiveBytes:buffer maxLength:sizeof(buffer) additionally:YES error:errorPtr] == -1) {
			return NO;
		}
		
		if (strncmp((char *)buffer + 12, "OK", 2)) {
			if (n++ < kOpeningRetryCount) {
				[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:kOpeningRetryInterval]]; 
				continue;
			}
			if (errorPtr != NULL) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The AIR-EDGE PHONE wasn't the top screen.", @"error.notTopScreen.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:WXKErrorDomain
												code:WXKPhoneNotReadyError
											userInfo:userInfo];
			}
			return NO;
			//			[NSException raise:kAirHPhoneException format:@"The AIR-EDGE PHONE wasn't the top screen."];
		}
		break;
	}
	loggedin = YES;
	
	// convert password to C-string
	NSData *passwordData = [password dataUsingEncoding:NSShiftJISStringEncoding];
	if(passwordData == nil) {
		// in case security code contains characters which can be converted to SJIS encoding
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The security code contains incorrect characters.", @"error.passswordStringEncoding.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneInapplicablePasswordStringEncodingError
										userInfo:userInfo];
		}
		return NO;
	}
	int passwordDataLength = [passwordData length];
	if(passwordDataLength <= 0 || passwordDataLength > 4) {
		// in case password length is 0 or longer than 4
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The security code must consist of just 4 characters.", @"error.passswordStringLength.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneInapplicablePasswordStringLengthError
										userInfo:userInfo];
		}
		return NO;
	}
	[passwordData getBytes:buffer length:passwordDataLength];
	buffer[passwordDataLength] = 0;
	
	// send password to cellphone
	if ([self sendCommand:1 subCommand:0x0201 parameter:(char *)buffer error:errorPtr] == NO) {
		return NO;
	}
	if ([self receiveBytes:buffer maxLength:sizeof(buffer) additionally:YES error:errorPtr] == -1) {
		return NO;
	}
	if(strncmp((char *)buffer + 12, "OK", 2)) {
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Security code is incorrect.", @"error.incorrectPassword.description") 
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:WXKErrorDomain
											code:WXKPhoneIncorrectPasswordError
										userInfo:userInfo];
		}
		return NO;
	}
	return YES;
}

// --- logout

- (BOOL)logoutError:(NSError **)error
{
	if(loggedin) {
		if ([self sendCommand:0xff subCommand:0x0102 parameter:"" error:error] == NO) {
			return NO;
		}
		loggedin = NO;
		return YES;
	} else {
		if (error != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The application tried to log out but cellphone had not been logged in.", @"error.notLoggedIn.description") 
																 forKey:NSLocalizedDescriptionKey];
			*error = [NSError errorWithDomain:WXKErrorDomain
										 code:WXKPhoneNotLoggedInError
									 userInfo:userInfo];
		}
		return NO;
	}
}

@end